<?php

namespace App\Http\Requests\Web;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class RoomReservationRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     * So the user must be guest
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $namesRegex = '/^([a-zA-Z ]|[\p{Arabic}])*$/u';
        $phonePattern = '/^(009665|9665|\+9665|05|5)(5|0|3|6|4|9|1|8|7)([0-9]{7})$/';

        $rules = [
            'name' => ['required', 'min:3', 'max:100', "regex:$namesRegex"],
            'email' => ['required', 'email', 'max:100'],
            'phone' => ['required', 'numeric', 'regex:'.$phonePattern],
            'id_image' => ['required', 'mimes:jpg,jpeg,png'],
            'branch_id' => ['required', 'exists:branches,id'],
            'room_type_id' => ['required', 'exists:room_types,id'],
            'room_id' => ['required', 'exists:rooms,id'],
            'start_date' => ['required', 'date', 'after_or_equal:'.date('Y-m-d')],
            'end_date' => ['required', 'date', 'after:start_date'],
            'pay_way' => ['required', 'in:2,6,11,14'],

        ];
        return $rules;
    }

    public function attributes()
    {
        $field = metaFields('roomReservations');
        $local = getCurrentLocale();
        $attributes = [
            'name' => $field[$local]['name'] ?? __('roomReservations.name'),
            'email' => $field[$local]['email'] ?? __('roomReservations.email'),
            'phone' => $field[$local]['phone'] ?? __('roomReservations.phone'),
            'id_image' => $field[$local]['id_image'] ?? __('roomReservations.id_image'),
            'branch_id' => $field[$local]['branch_id'] ?? __('roomReservations.branch_id'),
            'room_type_id' => $field[$local]['room_type_id'] ?? __('roomReservations.room_type_id'),
            'room_id' => $field[$local]['room_id'] ?? __('roomReservations.room_id'),
            'start_date' => $field[$local]['start_date'] ?? __('roomReservations.start_date'),
            'end_date' => $field[$local]['end_date'] ?? __('roomReservations.end_date'),
            'pay_way' => $field[$local]['pay_way'] ?? __('roomReservations.pay_way'),
        ];

        return $attributes;
    }


    public function messages()
    {
        return [
            'start_date.after_or_equal' => __('roomReservations.start_date_validation'),
            'end_date.after' => __('roomReservations.end_date_validation'),
        ];
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests

        if(Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
