<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed email
 * @property mixed message
 * @property mixed visitor_email
 * @property mixed reply
 * @property mixed name
 * @property mixed subject
 * @property mixed visitorMessageDetail
 */
class VisitorMessageRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (Auth::guest()) {
            return true;
        }
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'visitorMessages' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [];
        if ($this->visitorMessageDetail){
            foreach (languages() as $lang) {
                $rules['header_image'] = 'required';
                $lang_rules=[
                    'header_title_'.$lang->local=> 'required|min:3|max:100',
                    'title_'.$lang->local=> 'required|min:3|max:100',
                ];
                $rules= array_merge($rules, $lang_rules);
            }
        } else {
            if (Auth::user() && Auth::user()->role){
                $admin_rules = [
                    'visitor_email' => 'required|email',
                    'reply' => 'required|min:10',
                ];

                $rules = array_merge($rules, $admin_rules);

            }
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('visitorMessages');
        $local = getCurrentLocale();
        $attributes = [
            'name' => $field[$local]['name'] ?? __('visitorMessages.name'),
            'email' => $field[$local]['email'] ?? __('visitorMessages.email'),
            'subject' => $field[$local]['subject'] ?? __('visitorMessages.subject'),
            'message' => $field[$local]['message'] ?? __('visitorMessages.message'),
            'visitor_email' => $field[$local]['visitor_email'] ?? __('visitorMessages.visitor_email'),
            'reply' => $field[$local]['reply'] ?? __('visitorMessages.reply'),
            'header_image' => $field[$local]['header_image'] ?? __('visitorMessages.header_image'),
        ];

        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';

            $lang_attributes=[
                'header_title_'.$lang->local => ($field[$lang->local]['header_title'] ?? __('visitorMessages.header_title')) . $local,
                'title_'.$lang->local=> ($field[$lang->local]['title'] ?? __('addValues.title')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }

        return $attributes;
    }
    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax purchases like API purchases
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
