<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_type
 * @property mixed video_url
 * @property mixed images
 * @property mixed valueDetail
 */
class SiteSliderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('siteSliders'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {

        $rules = [
            'button_url' => 'nullable|url|max:300'
        ];

        foreach (languages() as $lang) {
            $lang_rules=[
                "title_$lang->local" => 'required|min:3|max:100',
                "button_title_$lang->local" => 'required_with:button_url|nullable|min:3|max:100',
                "description_$lang->local" => 'required|min:3|max:65000',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        $rules['image'] = 'nullable';

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('siteSliders');
        $local = getCurrentLocale();
        $attributes=[
            'image' => $field[$local]['image'] ?? __('siteSliders.image'),
            'button_uel' => $field[$local]['button_uel'] ?? __('siteSliders.button_uel'),
        ];


        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';
            $lang_attributes=[
                'title_'.$lang->local => ($field[$lang->local]['title'] ?? __('siteSliders.title')) . $local,
                'button_title_'.$lang->local => ($field[$lang->local]['button_title'] ?? __('siteSliders.button_title')) . $local,
                'description_'.$lang->local => ($field[$lang->local]['description'] ?? __('siteSliders.description')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    public function messages()
    {
        $attributes = [];
        foreach (languages() as $lang) {
            $lang_attributes=[
                "button_title_$lang->local.required_with" => __("siteSliders.button_title_{$lang->local}_required"),
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::ajax()){
            if ($validator->fails()){
                $errors = $validator->errors()->first();
                throw new HttpResponseException(response()->json(['errors' => $errors]));
            }

        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
