<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed symbol
 */
class PageRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(Auth::user()->role == 'admin' )
        {
            return true;
        }

        if(Auth::user()->hasModule( 'pages' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [];

        $linkPattern = '/^(https:\/\/(www\.)?(youtu|youtube){1}.([a-z]{1,4}))(\/watch\?v=|\/embed\/|\/){0,1}([a-zA-z0-9]{2,15})(&.)*/';

        if (in_array($this->page->id, [1])){
            $rules['video_added_type'] = 'required|in:url,video';
        }

        foreach (languages() as $lang) {
            $lang_rules=[
                'page_title_'.$lang->local=> 'required|min:3|max:50',
            ];

            if (in_array($this->page->id, [1,2,3,4,5,6])){
                $lang_rules['title_'.$lang->local] = 'required|min:3|max:100';
            }

            $lang_rules['description_'.$lang->local] = 'required|min:3';

            $rules= array_merge($rules, $lang_rules);

            if (in_array($this->page->id, [8])){
                $rules["users_$lang->local"] = 'required|min:3|max:100';
                $rules["services_$lang->local"] = 'required|min:3|max:100';
                $rules["employees_$lang->local"] = 'required|min:3|max:100';
                $rules["projects_$lang->local"] = 'required|min:3|max:100';
            }

        }

        if (in_array($this->page->id, [8])){
            $rules['statistics.*'] = 'required|string|min:1|max:30';
        }

//        foreach (languages() as $lang) {
//            $lang_rules = [];
//            if (in_array($this->page->id, [1])){
//                $lang_rules['point_1_title_'.$lang->local] = 'required|min:3|max:100';
//                $lang_rules['point_1_desc_'.$lang->local] = 'required|min:3|max:200';
//                $lang_rules['point_2_title_'.$lang->local] = 'required|min:3|max:100';
//                $lang_rules['point_2_desc_'.$lang->local] = 'required|min:3|max:200';
//            }
//
//            $rules= array_merge($rules, $lang_rules);
//        }

        if (in_array($this->page->id, [1,2,3,4,5,6,7,8])){
            $rules['image'] = 'required';
        }

        if (in_array($this->page->id, [1])){
            $rules['video_url'] = ['required_if:video_added_type,url', 'nullable', 'regex:'.$linkPattern];
            $rules['video'] = ['required_if:video_added_type,video'];
        }


//        if (in_array($this->page->id, [1])){
//            $rules['about_image'] = 'required';
//            $rules['home_page_image'] = 'required';
//            $rules['point_1_image'] = 'required';
//            $rules['point_2_image'] = 'required';
////            $rules['images'] = 'required|array|min:2|max:2';
//        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('pages');
        $local = getCurrentLocale();
        $attributes=[
            'image' => $field[$local]['image'] ?? __('pages.image'),
            'video_url' => $field[$local]['video_url'] ?? __('pages.video_url'),
            'video' => $field[$local]['video'] ?? __('pages.video'),
            'statistics.*' => $field[$local]['statistics'] ?? __('pages.statistics'),
            'video_added_type' => $field[$local]['video_added_type'] ?? __('pages.video_added_type'),
//            'about_image' => $field[$local]['about_image'] ?? __('pages.about_image'),
//            'home_page_image' => $field[$local]['home_page_image'] ?? __('pages.home_page_image'),
//            'point_1_image' => $field[$local]['point_1_image'] ?? __('pages.point_1_image'),
//            'point_2_image' => $field[$local]['point_2_image'] ?? __('pages.point_2_image'),

        ];
        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';
            $lang_attributes=[
                'page_title_'.$lang->local => ($field[$lang->local]['page_title'] ?? __('pages.page_title')) . $local,
                'title_'.$lang->local => ($field[$lang->local]['title'] ?? __('pages.title')) . $local,
                'users_'.$lang->local=> ($field[$lang->local]['users'] ?? __('pages.users')).$local,
                'services_'.$lang->local=> ($field[$lang->local]['services'] ?? __('pages.services')).$local,
                'employees_'.$lang->local=> ($field[$lang->local]['employees'] ?? __('pages.employees')).$local,
                'projects_'.$lang->local=> ($field[$lang->local]['projects'] ?? __('pages.projects')).$local,
                'description_'.$lang->local=> ($field[$lang->local]['description'] ?? __('pages.description')).$local,

            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }

    public function messages()
    {
        return [
            'video_url.required_if' => __('pages.video_url_validation'),
            'video.required_if' => __('pages.video_validation'),
        ];
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
