<?php

namespace App\Http\Requests;

use App\Service;
use Illuminate\Support\Carbon;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed type
 * @property mixed video_url
 * @property mixed image
 * @property mixed file
 * @property mixed appointment_date
 */
class AppointmentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('appointments'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {




        $rules = [
            'service_id' => 'required|exists:services,id',
            'doctor_id' => 'required|exists:doctors,id',
            'appointment_date' => 'required|date|after_or_equal:'.date('Y-m-d'),
            'doctor_start_shift' => ['required'],
            'doctor_end_shift' => ['required', 'after:doctor_start_shift'],
        ];

        if ($this->request->get('service_id') && $this->request->get('doctor_start_shift') && $this->request->get('doctor_end_shift')){
            $service = Service::withDescription([$this->request->get('service_id')])->first();
            $duration = $service->durations;
            $this->request->add(['durations' => $duration]);

            $startTime = Carbon::make($this->request->get('doctor_start_shift'));
            $endTime = Carbon::make($this->request->get('doctor_end_shift'));

            $diff = $endTime->diffInMinutes($startTime);

            if ($diff < $duration){
                $rules['doctor_end_shift_validation'] = 'required';
            }
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('appointments');
        $local = getCurrentLocale();
        $attributes=[
            'service_id' => $field[$local]['service_id'] ?? __('appointments.service_id'),
            'doctor_id' => $field[$local]['doctor_id'] ?? __('appointments.doctor_id'),
            'appointment_date' => $field[$local]['appointment_date'] ?? __('appointments.appointment_date'),
            'doctor_start_shift' => $field[$local]['doctor_start_shift'] ?? __('appointments.doctor_start_shift'),
            'doctor_end_shift' => $field[$local]['doctor_end_shift'] ?? __('appointments.doctor_end_shift'),
            'doctor_end_shift_validation' => $field[$local]['doctor_end_shift_validation'] ?? __('appointments.doctor_end_shift_validation'),
        ];

        return $attributes;
    }


    public function messages()
    {
        $attributes = [
            'appointment_date.after_or_equal' => __('appointments.appointment_date_validation'),
            'doctor_end_shift.after' => __('appointments.doctor_end_shift_validation') . $this->request->get('durations') . __('appointments.minutes'),
            'doctor_end_shift_validation.required' => __('appointments.doctor_end_shift_validation') . $this->request->get('durations') . __('appointments.minutes'),
        ];

        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }}
