<?php

namespace App\Http\Requests\API;

use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed password
 * @property mixed email
 * @property mixed photo
 */
class SavePaymentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'invoice_id'=> 'required|exists:airports,id|different:arrival_airport',
            'arrival_airport'=> 'required|exists:airports,id|different:departure_airport',
            'departure_city'=> 'required|exists:cities,id|different:arrival_city',
            'arrival_city'=> 'required|exists:cities,id|different:departure_city',
            'trip_type'=> 'required|in:go,go_back',
            'going_date'=> 'required|date|after_or_equal:now',
            'return_date'=> 'required|date|after_or_equal:going_date',
            'class_type'=> 'required|in:economic,distinctive_economy,business_class,first_class',
            'trip_direction'=> 'required|in:direct,transit',
            'children_count'=> 'required|numeric|gt:0',
            'adult_count'=> 'required|numeric|gt:0',

        ];

        if (request('guest')){
            $rules['name'] = 'required|min:3|max:100';
            $rules['email'] = 'required|email';
            $rules['code'] = 'required|exists:countries,code';
            $rules['phone'] = 'required|numeric|regex:'.$phonePattern;
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $attributes= [
            'departure_airport'=> __('flights.departure_airport'),
            'arrival_airport'=> __('flights.arrival_airport'),
            'departure_city'=> __('flights.departure_city'),
            'arrival_city'=> __('flights.arrival_city'),
            'trip_type'=> __('flights.trip_type'),
            'going_date'=> __('flights.going_date'),
            'return_date'=> __('flights.return_date'),
            'class_type'=> __('flights.class_type'),
            'trip_direction'=> __('flights.trip_direction'),
            'children_count'=> __('flights.children_count'),
            'adult_count'=> __('flights.adult_count'),
            'name'=> __('flights.name'),
            'email'=> __('flights.email'),
            'code'=> __('flights.code'),
            'phone'=> __('flights.phone'),
        ];
        return $attributes;
    }

    public function messages()
    {
        return [
            'children_count.gt' => __('flights.children_count_gt'),
            'adult_count.gt' => __('flights.adult_count_gt'),
            'departure_airport.different' => __('flights.departure_airport_different'),
            'arrival_airport.different' => __('flights.arrival_airport_different'),
            'departure_city.different' => __('flights.departure_city_different'),
            'arrival_city.different' => __('flights.arrival_city_different'),
            'going_date.after_or_equal' => __('flights.going_date_after_or_equal'),
            'return_date.after_or_equal' => __('flights.return_date_after_or_equal'),
        ];
    }


    /**
     * Handle a failed validation attempt.
     * Override the parent method action if the request is for AJAX
     * @param Validator $validator
     * @return void
     */
    protected function failedValidation(Validator $validator) :void
    {
        // Change response attitude if the request done via Ajax requests like API requests
        $errors = (new ValidationException($validator))->errors();
        $message = $validator->errors()->first();
        throw new HttpResponseException(response()->json(['message'=> $message, 'errors' => $errors],  400));
    }
}
