<?php

namespace App\Http\Controllers\Web;

use App\Branch;
use App\Helpers\ImageHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Web\RoomComplaintRequest;
use App\Http\Requests\Web\RoomEvaluateRequest;
use App\Http\Requests\Web\RoomReservationRequest;
use App\Room;
use App\RoomComplaint;
use App\RoomEvaluate;
use App\RoomReservation;
use App\RoomService;
use App\RoomType;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\View\View;


class RoomController extends Controller
{
    private $view = 'web.rooms.';

    public function __construct()
    {
        $this->middleware('check_user_operating_system');
    }

    /**
     * Show the application dashboard.
     * @param Request $request
     * @return View
     */
    public function index(Request $request): View
    {
        $title = __('rooms.head');
        $sessionBranchId = session('branch_id');
        $sessionRoomTypeId = session('room_type_id');
        $sessionPersonCount = session('person_count');
        $sessionRate = session('rate');



        $branches = Branch::withDescription();
        $roomTypes = $sessionRoomTypeId ? RoomType::withDescription($sessionRoomTypeId) : [];


        $rooms = self::getRooms(
            ($sessionBranchId ?? $request->branch_id),
            ($sessionRoomTypeId ?? $request->room_type_id),
            ($sessionPersonCount ?? $request->person_count),
            ($sessionRate ?? $request->rate)
        )->paginate(6);

        $personCounts = settings('max_room_person_count') ?? 4;


        if ($request->ajax()){
            session()->put([
                'branch_id' => $request->branch_id,
                'room_type_id' => $request->room_type_id,
                'person_count' => $request->person_count,
                'rate' => $request->rate,
            ]);

            $rooms = self::getRooms(
                $request->branch_id,
                $request->room_type_id,
                $request->person_count,
                $request->rate
            )->paginate(6);

            return view($this->view.'search_result', get_defined_vars());
        }



        return view($this->view.'index', get_defined_vars());
    }

    public function show(Room $room)
    {
        $room = Room::withDescription($room->id);
        $title = __('rooms.description');
        $ratingCounts = RoomEvaluate::where('room_id', $room->id)->count();
        $ratingLevel = null;
        switch ($room->rate){
            case $room->rate <= 1:
                $ratingLevel = __('rooms.bad');
                break;
            case $room->rate <= 2:
                $ratingLevel = __('rooms.acceptable');
                break;
            case $room->rate <= 3:
                $ratingLevel = __('rooms.good');
                break;
            case $room->rate <= 4:
                $ratingLevel = __('rooms.very_good');
                break;
            case $room->rate <= 5:
                $ratingLevel = __('rooms.excellent');
                break;
        }
        $roomServices = is_array($room->room_services_ids) && count($room->room_services_ids) ? RoomService::withDescription($room->room_services_ids) : [];

        $roomReviews = RoomEvaluate::withDescription(null, $room->id, 1);

        return view($this->view.'show', get_defined_vars());
    }

    public function roomEvaluate(RoomEvaluateRequest $request)
    {
        $data = $request->all();
        $room = Room::find($request->room_id);
        $data['branch_id'] = $room->branch_id;
        RoomEvaluate::create($data);

        $roomRate = RoomEvaluate::where('room_id', $room->id)
            ->selectSub("SELECT AVG(`reception_evaluation`) FROM `room_evaluates` where room_id = {$room->id} LIMIT 1", 'reception_evaluation')
            ->selectSub("SELECT AVG(`safety_evaluation`)FROM `room_evaluates` where room_id = {$room->id}  LIMIT 1", 'safety_evaluation')
            ->selectSub("SELECT AVG(`room_evaluation`) FROM `room_evaluates` where room_id = {$room->id}  LIMIT 1", 'room_evaluation')
            ->selectSub("SELECT AVG(`food_evaluation`) FROM `room_evaluates` where room_id = {$room->id}  LIMIT 1", 'food_evaluation')->first();

        $newRate = ($roomRate->reception_evaluation+$roomRate->safety_evaluation+$roomRate->room_evaluation+$roomRate->food_evaluation)/4;
        $room->update(['rate' => $newRate]);

        return response()->json(['message' => __('roomEvaluates.evaluate_success')]);

    }

    public function roomComplaint(RoomComplaintRequest $request)
    {
        $data = $request->all();
        $room = Room::find($request->room_id);
        $data['branch_id'] = $room->branch_id;
        $data['room_type_id'] = $room->room_type_id;
        $data['image'] = $room->image ? ImageHelper::upload($request->image, 'room_complaints') : null;
        RoomComplaint::create($data);
        return response()->json(['message' => __('roomComplaints.complaint_success')]);

    }

    public function reservationForm()
    {
        $title = __('main.reservation_request');
        $branches = Branch::withDescription();
        $showApplePay = request('operating_system') == 'APPLE' ? true : false;

        return view($this->view.'reservation_form', get_defined_vars());

    }

    public function roomReservationForm(Room $room)
    {

        if (!$room->room_status){
            return back()->with('error_message', __('rooms.room_unavailable'));
        }

        $title = __('main.reservation_request');
        $branches = Branch::withDescription();
        $showApplePay = request('operating_system') == 'APPLE' ? true : false;

        return view($this->view.'room_reservation_form', get_defined_vars());

    }

    public function roomReservation(RoomReservationRequest $request)
    {
        if ($request->ajax()){

            $room = Room::find($request->room_id);
            if (!$room){
                return response()->json(['errors' => __('rooms.no_data')]);
            }

            if (!$room->room_status){
                return response()->json(['errors' =>  __('rooms.room_unavailable')]);
            }

            $showApplePay = request('operating_system') == 'APPLE' ? true : false;

            if (!$showApplePay && $request->pay_way == 11){
                return response()->json(['errors' =>  __('rooms.payment_not_allowed')]);
            }


            $data = $request->all();
            $startDate = Carbon::make($request->start_date);
            $endDate = Carbon::make($request->end_date);
            $daysCount = $endDate->diffInDays($startDate);
            $paymentWay = [
                2 => 'visa_master',
                6 => 'mada',
                11 => 'apple_pay',
                14 => 'stc_pay',
                4 => 'sadad',
                20 => 'visa_master_direct',
                9 => 'visa_master_direct_3ds_flow',
                5 => 'benefit',
                8 => 'uae_debit_cards',
                1 => 'knet',
                3 => 'amex',
                7 => 'qatar_debit_cards'
            ];


            $total = $daysCount*$room->room_price;
            $data['days_count'] = $daysCount;
            $data['total'] = $total;
            $data['id_image'] = ImageHelper::upload($request->id_image, 'users_id_images');
            $data['pay_way'] = $paymentWay[$request->pay_way];
            $data['start_date'] = date('Y-m-d', strtotime($request->start_date));
            $data['end_date'] = date('Y-m-d', strtotime($request->end_date));
            $branchName = descriptions(Branch::find($request->branch_id))->name;
            $roomName = descriptions(Room::find($request->room_id))->name;
            $roomType = descriptions(RoomType::find($request->room_type_id))->name;

            $roomReservation = RoomReservation::create($data);
            $sessionData = ['room_reservation' => $roomReservation];
            $sessionData['room_reservation']['pay_way_api'] = $request->pay_way;
            $sessionData['room_reservation']['branch_name'] = $branchName;
            $sessionData['room_reservation']['room_name'] = $roomName;
            $sessionData['room_reservation']['room_type'] = $roomType;

            session()->put($sessionData);

            return MyFatoorah::pay();

        }
    }

    public function getRoomTypes(Request $request)
    {
        if ($request->ajax()){
            $roomTypes = RoomType::withDescription(null, $request->branch_id);
            return response()->json(['room_types' => $roomTypes]);
        }
    }

    public function getRoomsByType(Request $request)
    {
        if ($request->ajax()){
            $rooms = Room::withDescription(null, $request->room_type_id);
            return view($this->view.'get_rooms', get_defined_vars());
        }
    }

    public static function getRooms($branchId = null, $roomTypeId = null, $personCount = null, $rate = null)
    {
        $langId = currentLanguage()->id;
        $query = Room::orderByDesc('rooms.id')
            ->join('room_descriptions as rDesc', 'rDesc.room_id', 'rooms.id')
            ->where('rDesc.language_id', $langId)

            ->leftJoin('room_types', 'room_types.id', 'rooms.room_type_id')
            ->leftJoin('room_type_descriptions as rtDesc', 'rtDesc.room_type_id', 'room_types.id')
            ->where('rtDesc.language_id', $langId)

            ->leftJoin('branches', 'branches.id', 'rooms.branch_id')
            ->leftJoin('branch_descriptions as bDesc', 'bDesc.branch_id', 'branches.id')
            ->where('bDesc.language_id', $langId);

        if ($roomTypeId){
            $query->where('rooms.room_type_id', $roomTypeId);
        }

        if ($branchId){
            $query->where('rooms.branch_id', $branchId);
        }

        if ($personCount){
            $query->where('rooms.person_count', $personCount);
        }

        if ($rate){
            $query->where('rooms.rate', '>=', $rate);
        }


        $query->select([
            'rooms.id',
            'rooms.rate',
            'rooms.image',
            'rooms.room_type_id',
            'rooms.branch_id',
            'rDesc.name',
            'rDesc.description',
            'rtDesc.name as roomTypeName',
            'bDesc.name as branchName',
        ])
            ->selectSub("SELECT AVG(`reception_evaluation`) FROM `room_evaluates` where room_evaluates.id = rooms.id LIMIT 1", 'reception_evaluation')
            ->selectSub("SELECT AVG(`safety_evaluation`)FROM `room_evaluates` where room_evaluates.id = rooms.id LIMIT 1", 'safety_evaluation')
            ->selectSub("SELECT AVG(`room_evaluation`) FROM `room_evaluates` where room_evaluates.id = rooms.id LIMIT 1", 'room_evaluation')
            ->selectSub("SELECT AVG(`food_evaluation`) FROM `room_evaluates` where room_evaluates.id = rooms.id LIMIT 1", 'food_evaluation');

        return $query;

    }


}
