<?php

namespace App\Http\Controllers\Web;

use Illuminate\Support\Facades\Session;
use App\Appointment;
use App\AppointmentTime;
use App\Branch;
use App\Doctor;
use App\Evaluate;
use App\Helpers\Sms;
use App\Helpers\GoogleCalenderApi;
use App\Http\Controllers\Controller;
use App\Http\Requests\Web\AddBookingRequest;
use App\Http\Requests\Web\AddEvaluateRequest;
use App\Jobs\SendBookingMessageJob;
use App\Reservation;
use App\Service;
use App\User;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class BookingController extends Controller
{

    use ThrottlesLogins;

    /**
     * The maximum number of attempts to allow.
     *
     * @return int
     */
    protected $maxAttempts = 3;


    /**
     * The number of minutes to throttle for.
     *
     * @return int
     */
    protected $decayMinutes = 1;


    /**
     * The field name throttle trait will use for.
     *
     * @return string
     */
    public function username() : string
    {
        return 'username';
    }

    private $view = 'web.bookings.';

    /**
     * Show the application dashboard.
     * @param Service $service
     * @return View
     */
    public function calendar(Reservation $reservation)
    {
        Session::put('reservation_id', $reservation->id);
        $urlRedirect = GoogleCalenderApi::googleOauthURL();
        return redirect($urlRedirect);
    }

    public function bookingForm(Service $service): View
    {
        $title = __('services.appointment_request');
        $service = Service::withDescription([$service->id])->first();
        $branches = Branch::withDescription($service->branch_ids);
        $doctors = Doctor::join('doctor_descriptions as doctorDesc', 'doctorDesc.doctor_id', 'doctors.id')
            ->where('doctorDesc.language_id', currentLanguage()->id)
            ->whereJsonContains('service_ids', "$service->id")
            ->join('appointments', 'appointments.doctor_id', 'doctors.id')
            ->whereNull('appointments.deleted_at')
            ->where('appointments.service_id', $service->id)
            ->select([
                'doctors.id',
                'doctors.service_ids',
                'doctors.image',
                'doctorDesc.name',
                'doctorDesc.specialist',
            ])->cursor();
//        $availableDates = Appointment::where('appointment_date', '>=', date('Y-m-d'))->pluck('appointment_date')->toArray();
//        $availableDates = array_unique($availableDates);
        return view($this->view.'form', get_defined_vars());
    }

    public function addBooking(AddBookingRequest $request, Service $service)
    {
        $verifyCode = $this->verifyCode($request);
        if($verifyCode->original['errors']){
            return $verifyCode;
        }

        $user = User::where('verify_code', $request->verify_code)
            ->where('phone', $request->phone)
            ->where('email', $request->email)->first();

        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

        $user->update(['verify_code' => null]);

        $data = $request->all();
        $data['service_id'] = $service->id;
        $data['durations'] = $service->durations;
        $data['doctor_id'] = $request->doctor_id ? $request->doctor_id : $request->rand_doctor_id;
        $appointmentTime = AppointmentTime::find($request->appointment_time);
        if (!$appointmentTime){
            return response()->json(['errors' => __('reservations.no_time_available')]);
        }
        $data['appointment_time'] = $appointmentTime->available_time;

        $token = time().'_'.str_random();
        $token = md5($token);
        $data['evaluate_token'] = $token;
        $reservation = Reservation::create($data);
        $appointmentTime->update(['status' => 'not_available']);

        $branch = Branch::find($request->branch_id);
        $doctor = Doctor::find($data['doctor_id']);

        $reservationDetails = [
            'id' => $reservation->id,
            'service_name' => $service->currentDescription->name,
            'service_duration' => $service->durations,
            'branch_name' => $branch->currentDescription->name,
            'branch_address' => $branch->currentDescription->address,
            'branch_phone' => $branch->phone,
            'branch_map' => $branch->map,
            'doctor_name' => $doctor->currentDescription->name,
            'appointment_date' => date('F, d, D, Y', strtotime($request->appointment_date)) . ' at ' . date('h:i A', strtotime($request->appointment_time)),
        ];

        session()->put(['reservation_details' => $reservationDetails]);

//        $phone = explode('0', $user->phone, 2);
        $phone = $user->phone;
        $serviceName = $reservationDetails['service_name'];
        $doctorName = $reservationDetails['doctor_name'];
        $appointmentDate = $reservationDetails['appointment_date'];

        $message = __('reservations.appointment_msg_status_1');
        $message = str_replace_array('@@', [$serviceName, $doctorName, $appointmentDate], $message);
//        try{
//            Sms::send($phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        try{
            $data = [
                'subject' => __('reservations.reservation_confirmation_msg'),
                'message' => $message,
                'email' => $request->email
            ];
            SendBookingMessageJob::dispatch($data, 'web.bookings.emails.send_email');

        } catch (\Exception $e){}

        session()->flash('process_done', true);

        return response()->json(['url' => route('booking.bookingDetails'),'message' => __('reservations.reservation_added_successfully')]);

    }

    public function bookingDetails()
    {
        if (!session('reservation_details')){
            return redirect()->route('home');
        }

        $title = __('reservations.edit');
        $reservation = session('reservation_details');
        return view($this->view.'reservation_details', get_defined_vars());
    }

    public function evaluate(Reservation $booking, $token)
    {
        if ($token != $booking->evaluate_token){
            return redirect()->route('home');
        }

        $isEvaluateBefore = Evaluate::where('reservation_id', $booking->id)->first();

        if (($isEvaluateBefore->id ?? null)){
            return redirect()->route('home');
        }
        $title = __('reservations.evaluate');
        $service = Service::withDescription([$booking->service_id])->first();
        $serviceName = ($service->id ?? null) ? $service->name : null;
        return view($this->view.'evaluate_form', get_defined_vars());
    }

    public function addEvaluate(AddEvaluateRequest $request, Reservation $booking, $token)
    {
        if ($token != $booking->evaluate_token){
            return response()->json(['errors' => __('evaluates.evaluate_not_allow')]);
        }

        $isEvaluateBefore = Evaluate::where('reservation_id', $booking->id)->first();

        if (($isEvaluateBefore->id ?? null)){
            return response()->json(['errors' => __('evaluates.evaluate_before')]);
        }

        Evaluate::create([
            'name' => $booking->name,
            'reservation_id' => $booking->id,
            'service_id' => $booking->service_id,
            'appointment_date' => $booking->appointment_date,
            'appointment_time' => $booking->appointment_time,
            'rate' => $request->rate,
            'comment' => $request->comment,
        ]);

        return response()->json(['message' => __('evaluates.evaluate_added_successfully')]);

    }

    public function getAppointmentDatesAndTimes(Request $request)
    {
        if($request->ajax()){

            // When the user clicks on the appointment date
            // to get the available times on that date
            if ($request->get_available_times){

                $appointment = Appointment::where('service_id', $request->service_id)
                    ->where('appointment_date', $request->appointment_date)->first();

                if (!$appointment->id){
                    return response()->json(['errors' => __('reservations.no_dates_available')]);
                }

                $appointmentTimes = AppointmentTime::where('appointment_id', $appointment->id)
                    ->where('status', 'available')->get()->groupBy('available_time')->toArray();

                if (!count($appointmentTimes)){
                    return response()->json(['errors' => __('reservations.no_time_available')]);
                }

                $timesAm = [];
                $timesPm = [];
                $timesAmIds = [];
                $timesPmIds = [];

                foreach ($appointmentTimes as $time => $item) {
                    $timeAmOrPm = date('h:i A', strtotime($time));
                    if (strpos($timeAmOrPm, 'AM')){
                        $timesAm[] = $timeAmOrPm;
                        $timesAmIds[] = $item[0]['id'];
                    }
                    if (strpos($timeAmOrPm, 'PM')){
                        $timesPm[] = $timeAmOrPm;
                        $timesPmIds[] = $item[0]['id'];
                    }

                }

                return [
                    'available_time_am' => $timesAm,
                    'available_time_pm' => $timesPm,
                    'available_time_am_ids' => $timesAmIds,
                    'available_time_pm_ids' => $timesPmIds,
                ];
            }

            // Specific by user
            if ($request->specific_doctor_way == 1 && $request->doctor_id){
                $appointments = Appointment::where('service_id', $request->service_id)
                    ->where('doctor_id', $request->doctor_id)
                    ->where('appointment_date', '>=', date('Y-m-d'))->get();

                if (!count($appointments)){
                    return response()->json(['errors' => __('reservations.no_dates_available')]);
                }

                $doctor = Doctor::where('id', $request->doctor_id)->first();

                if (!$doctor){
                    return response()->json(['errors' => __('reservations.no_doctor_available')]);
                }

                $appointmentDates = $appointments->pluck('appointment_date')->toArray();

                return [
                    'appointment_date' => array_unique($appointmentDates),
                    'doctor_name' => $doctor->currentDescription->name,
                    'doctor_id' => $doctor->id,
                ];


            }

            // Specific by center
            if ($request->specific_doctor_way == 2 && !$request->doctor_id){

                $doctor = Doctor::whereJsonContains('service_ids', "$request->service_id")
                    ->join('appointments', 'appointments.doctor_id', 'doctors.id')
                    ->whereNull('appointments.deleted_at')
                    ->where('appointments.service_id', $request->service_id)
                    ->select(['doctors.id'])
                    ->inRandomOrder()->first();

                if (!$doctor){
                    return response()->json(['errors' => __('reservations.no_doctor_available')]);
                }


                $appointments = Appointment::where('service_id', $request->service_id)
                    ->where('doctor_id', $doctor->id)->get();


                if (!count($appointments)){
                    return response()->json(['errors' => __('reservations.no_dates_available')]);
                }

                $appointments = $appointments->pluck('appointment_date')->toArray();


                return [
                    'appointment_date' => $appointments,
                    'doctor_name' => $doctor->currentDescription->name,
                    'doctor_id' => $doctor->id,
                    'random_doctor_id' => $doctor->id,
                ];


            }
        }
    }

    public function verifyCode(Request $request)
    {

        $code = $request->verify_code;

        if (mb_strlen($code) < 4){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.enter_full_code')]);
        }

        $user = User::where('verify_code', $code)
            ->where('phone', $request->phone)
            ->where('email', $request->email)->first();

        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        } else {
            $user->update([
                'name' => $request->name,
//                'phone' => $request->phone,
            ]);
        }


        // Check the number of attempts to check the code
        if ($user->code_verify_attempt >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.verification_tries_exceed')]);
        }

        // Check if date less then today
        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'last_attempt_date' => date('Y-m-d')]);
        }

        // The user entered the wrong code and the number of attempts increased
        if (strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) && $user->verify_code != $code){
            $user->update(['code_verify_attempt' => $user->code_verify_attempt+1]);
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

//        $user->update(['verify_code' => null]);

        return response()->json(['errors' => false, 'message' => __('members.correct_code')]);
    }


    /**
     * check attempt try and generate code
     * @param $user
     * @return array
     */
    private function generateVerifyCode($user)
    {

        /**
         * It's commented for test only and after test
         * delete comment
         */

        if ($user->attempt_send_code >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return ['errors' => __('members.verification_code_exceed')];
        }

        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'attempt_send_code' => 0,'last_attempt_date' => date('Y-m-d')]);
        }

        $attempt_send_code = $user->attempt_send_code + 1;
        $last_attempt_date = date('Y-m-d');
        $code = rand(1000,9999);

//        session()->put(['verify_code' => $code]);

//        $phone = explode('0', $user->phone, 2);
        $phone = $user->phone;

        // This line is temporary and will be deleted later
        $code = in_array($phone, ['581524713', '543605073', '500000000']) ? '1234' : $code;

//        try{
//            $message = __('members.sms_message');
//            $message = str_replace('@@', $code, $message);
//            Sms::send($phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        /**
         * $data = [
        'user' => $user->name,
        'email' => $user->email,
        'code' => $code,
        ];
        CodeVerificationJob::dispatch($data, 'web.members.emails.verification_email');
         */

        $user->update(
            [
                'verify_code' => $code,
                'attempt_send_code' => $attempt_send_code,
                'last_attempt_date' => $last_attempt_date,
            ]
        );

        return ['errors' => false, 'code' => $code];
    }


    /**
     * Resend code for register and login
     * @param Request $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function sendCode(Request $request)
    {
        if ($request->ajax()){
            $phonePattern = '/^(5){1}([0-9]{8})$/';
            $validation = Validator::make($request->all(), [
                'phone' => ['required','regex:'.$phonePattern],
            ], [], ['phone' => __('reservations.phone')]);
            if ($validation->fails()){
                return response()->json([
                    'errors' => $validation->errors()->first(),
                ]);
            }

            $user = User::orWhere('phone', $request->phone)->orWhere('email', $request->email)->first();

            if (!$user){
                $user = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'phone' => $request->phone,
                    'role' => 'user',
                ]);
//                return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
            } else {
                $user->update(['email' => $request->email, 'phone' => $request->phone]);
            }

            if ($this->checkAttemptCount($request)){
                return $this->checkAttemptCount($request);
            }

            $generateVerifyCode = $this->generateVerifyCode($user);

            if ($generateVerifyCode['errors']){
                return $this->checkAttemptCount($request) ?? response()->json($generateVerifyCode);
            }

            return response()->json([
                'errors' => false,
                'success' => true,
                'message' => __('members.code_send_to_phone'),
                'code' => $generateVerifyCode['code']
            ]);
        }
    }


    /**
     * Check user attempts count
     * @param $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function checkAttemptCount($request)
    {
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            return $this->sendLockoutResponse($request);
        }
        $this->incrementLoginAttempts($request);

    }

    /**
     * Override parent sendLockoutResponse to handle it
     * to return json response
     * @param Request $request
     * @return JsonResponse
     */
    protected function sendLockoutResponse(Request $request)
    {
        $seconds = $this->limiter()->availableIn(
            $this->throttleKey($request)
        );
        return response()->json(['errors' => Lang::get('auth.throttle', [
            'seconds' => $seconds,
            'minutes' => ceil($seconds / 60),
        ])]);
    }

}
