<?php

namespace App\Http\Controllers\Admin;

use Exception;
use App\WebsitePhoto;
use App\WebsiteCategory;
use \Illuminate\View\View;
use Illuminate\Http\Request;
use App\WebsitePhotoDescriptions;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\WebsiteCategoryDescription;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\WebsiteCategoryRequest;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class WebsiteCategoryController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/websiteCategories';
    /** View folder */
    private $viewDirectory = 'admin.website_categories.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('websiteCategories.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the pages.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = WebsiteCategory::latest()
            ->join('website_category_descriptions AS gd', 'website_categories.id', 'gd.website_category_id')
            ->join('languages', 'languages.id', 'gd.language_id')
            ->where('local', LaravelLocalization::getCurrentLocale())
            ->select(['gd.name', 'website_categories.*']);

        if ($request->date_from) {
            $query->whereDate('website_categories.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('website_categories.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('gd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $galleries= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('galleries'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('websiteCategories.store');
        $head = metaFields('websiteCategories', 'new', getCurrentLocale()) ?? __('websiteCategories.new');
        $title = $head;
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param WebsiteCategoryRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(WebsiteCategoryRequest $request) :RedirectResponse
    {
        $gallery = WebsiteCategory::create();
        // Insert Description
        $this->saveData( $request, $gallery->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param WebsiteCategory $websiteCategory
     * @return View
     */
    public function edit(WebsiteCategory $websiteCategory) :View
    {
        $action= route('websiteCategories.update', $websiteCategory->id);
        $head = metaFields('websiteCategories', 'edit', getCurrentLocale()) ?? __('websiteCategories.edit');
        $title = $head;
        $query = WebsiteCategoryDescription::where('website_category_id', $websiteCategory->id)
            ->join('languages', 'languages.id', 'website_category_descriptions.language_id')
            ->select(['website_category_descriptions.*', 'languages.local']);

        $websiteCategoryDescription= $query->get();

        foreach ($websiteCategoryDescription as $row) {
            $websiteCategory[$row->local]= $row;
        }



        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param WebsiteCategoryRequest $request
     * @param WebsiteCategory $websiteCategory
     * @return RedirectResponse|Redirector
     */
    public function update(WebsiteCategoryRequest $request, WebsiteCategory $websiteCategory) :RedirectResponse
    {
        // Update the "updated_at" column only
        $websiteCategory->touch();

        // Delete old description
        WebsiteCategoryDescription::where('website_category_id', $websiteCategory->id)->delete();

        // delete old images
        WebsitePhoto::where('website_category_id', $websiteCategory->id)->delete();

        // Insert new description
        $this->saveData( $request, $websiteCategory->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param WebsiteCategoryRequest $request
     * @param int $website_category_id
     * @return void
     */

    private function saveData(WebsiteCategoryRequest $request, int $website_category_id ):void
    {
        $requestData= $request->all();

        $websiteCategoryDescription=[
            'name'=> $requestData['name_'.getCurrentLocale()],
            'slug'=> $requestData['slug_'.getCurrentLocale()],
            'website_category_id'=> $website_category_id,
            'language_id'=> currentLanguage()->id,
        ];

        WebsiteCategoryDescription::create($websiteCategoryDescription);

        if ($request->has('images')){
            foreach ($request->images as $key => $image) {
                $websitePhoto=[
                    'url'=> $requestData['image_url'][$key],
                    'image'=> $image,
                    'website_category_id'=> $website_category_id,
                ];

                $websitePhoto = WebsitePhoto::create($websitePhoto);

                $websitePhotoDescription = [
                    'title'=> $requestData['image_title_'.getCurrentLocale()][$key],
                    'website_photo_id'=> $websitePhoto->id,
                    'language_id'=> currentLanguage()->id,
                ];

                WebsitePhotoDescriptions::create($websitePhotoDescription);

            }
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $gallery = WebsiteCategory::withTrashed()->find($id);
        if ($gallery) {
            if ($gallery->deleted_at) {
                $gallery->forceDelete();
            } else {
                $gallery->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            WebsiteCategory::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            WebsiteCategory::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $gallery = WebsiteCategory::withTrashed()->find($id);

        if ($gallery){
            $gallery->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $gallery = WebsiteCategory::whereIn('id', $request->ids)->onlyTrashed();
        if ($gallery){
            $gallery->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
