<?php

namespace App\Http\Controllers\Admin;

/** Ajax Request */

use Exception;
use App\Tag;
use Illuminate\View\View;
use App\TagDescription;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use Illuminate\Http\RedirectResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\TagRequest; // for Validation
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class TagController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/tags';
    /** View folder */
    private $viewDirectory = 'admin.tags.';

    /**
     * Display a listing of the tags.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('tags.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the tags.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Tag::orderByDesc('tags.id')
        ->join('tag_descriptions AS cd', 'tags.id', 'cd.tag_id')
        ->where('cd.language_id', currentLanguage()->id)
        ->select(['cd.name', 'tags.*']);

        if ($request->date_from) {
            $query->whereDate('tags.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('tags.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('cd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ($request->tag_status) {
            $query->where('tags.status', 1);
        }

        if ($request->tag_status == '0') {
            $query->where('tags.status', '0');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }
        $tags= $query->paginate(100);

        return view($this->viewDirectory.'grid', get_defined_vars());
    }

    /**
     * Show the form for creating a new category.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('tags.store');
        $head = metaFields('tags', 'new', getCurrentLocale()) ?? __('tags.new');
        $title = $head;
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created category in storage.
     * @param TagRequest $request
     * @return JsonResponse
     */
    public function store(TagRequest $request) :JsonResponse
    {
        $tag = Tag::create($request->all());

        // Insert Description
        $this->saveData( $request, $tag->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('tags.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified category.
     *
     * @param  Tag  $tag
     * @return View
     */
    public function edit(Tag $tag) :View
    {
        $action= route('tags.update', $tag->id);
        $head = metaFields('tags', 'edit', getCurrentLocale()) ?? __('tags.edit');

        $query = TagDescription::where('tag_id', $tag->id)
        ->join('languages', 'languages.id', 'tag_descriptions.language_id')
        ->select(['tag_descriptions.*', 'languages.local']);

        $tagDescription= $query->get();

        foreach ($tagDescription as $row) {
            $tag[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars() );
    }

    /**
     * Update the specified category in storage.
     * @param TagRequest $request
     * @param Tag $tag
     * @return JsonResponse
     */
    public function update(TagRequest $request, Tag $tag) :JsonResponse
    {
        // Update the "updated_at" column only
        $tag->update($request->all());

        // Delete old description
        TagDescription::where('tag_id', $tag->id)->delete();

        // Insert new description
        $this->saveData( $request, $tag->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('tags.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param TagRequest $request
     * @param int $tag_id
     * @return void
     */

     private function saveData(TagRequest $request, int $tag_id ) :void
     {

        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'name'=> $requestData['name_'.$lang->local],
                'tag_id'=> $tag_id,
                'language_id'=> $lang->id,
            ];

            TagDescription::create($data);
        }

     }

    /**
     * Remove the specified category from storage.
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $tag= Tag::withTrashed()->find($id);
        if ($tag) {
            if ($tag->deleted_at) {
                $tag->forceDelete();
            } else {
                $tag->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several tags by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;

        if ($request->force) {
            Tag::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Tag::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $tag= Tag::withTrashed()->find($id);
        if ($tag) {
            $tag->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);

    }

    /**
     * Restore several tags by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        Tag::whereIn('id', $ids)->onlyTrashed()->restore();
        return response()->json(['message'=> __('dashboard.saveDone')]);
    }
}
