<?php

namespace App\Http\Controllers\Admin;

use App\Career;
use App\CareerApplication;
use App\Service;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class CareerApplicationController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/careerApplications';
    /** View folder */
    private $viewDirectory = 'admin.career_applications.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('careerApplications.head');
        $request= $request->toArray();
        $careers = Career::with('currentDescription')->whereHas('careerApplications')->get();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the careerApplications.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $langId = currentLanguage()->id;
        $query = CareerApplication::orderByDesc('career_applications.id')
            ->join('careers', 'career_applications.career_id', 'careers.id')
            ->join('career_descriptions AS cDesc', 'careers.id', 'cDesc.career_id')
            ->where('cDesc.language_id', $langId)

//            ->join('services', 'career_applications.service_id', 'services.id')
//            ->join('service_descriptions AS sDesc', 'services.id', 'sDesc.service_id')
//            ->where('sDesc.language_id', $langId)
            ->select(['cDesc.name as careerName', 'career_applications.*']);

        if ($request->submission_date_from) {
            $query->whereDate('career_applications.created_at', '>=', $request->submission_date_from);
        }

        if ($request->submission_date_to) {
            $query->whereDate('career_applications.created_at', '<=', $request->submission_date_to);
        }

        if ($request->name) {
            $query->where('career_applications.name', 'LIKE', '%'.$request->name.'%');
        }

//        if ($request->submission_date) {
//            $query->where('career_applications.created_at', $request->created_at);
//        }

        if ($request->career_id) {
            $query->where('career_applications.career_id', $request->career_id);
        }

        if ($request->email) {
            $query->where('career_applications.email', $request->email);
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $careerApplications= $query->paginate(100);

        return view($this->viewDirectory.'grid', get_defined_vars());
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param CareerApplication $careerApplication
     * @return View
     */
    public function edit(CareerApplication $careerApplication) :View
    {
        $head = metaFields('careerApplications', 'edit', getCurrentLocale()) ?? __('careerApplications.edit');
        $title = $head;
        $langId = currentLanguage()->id;
        $application = CareerApplication::where('career_applications.id', $careerApplication->id)
            ->join('careers', 'career_applications.career_id', 'careers.id')
            ->join('career_descriptions AS cDesc', 'careers.id', 'cDesc.career_id')
            ->where('cDesc.language_id', $langId)

//            ->join('services', 'career_applications.service_id', 'services.id')
//            ->join('service_descriptions AS sDesc', 'services.id', 'sDesc.service_id')
//            ->where('sDesc.language_id', $langId)
//
//            ->join('countries', 'career_applications.country_id', 'countries.id')
//            ->join('country_descriptions AS countryDesc', 'countries.id', 'countryDesc.country_id')
//            ->where('countryDesc.language_id', $langId)
//
//            ->join('educations', 'career_applications.education_id', 'educations.id')
//            ->join('education_descriptions AS edDesc', 'educations.id', 'edDesc.education_id')
//            ->where('edDesc.language_id', $langId)
            ->select([
                'cDesc.name as careerName',
//                'sDesc.name as specializationName',
//                'countryDesc.name as nationality',
//                'edDesc.name as education',
                'career_applications.*'
            ])->first();

        return view( $this->viewDirectory.'form', get_defined_vars());
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {

        $careerApplication = CareerApplication::withTrashed()->find($id);
        if ($careerApplication) {
            if ($careerApplication->deleted_at) {
                $careerApplication->forceDelete();
            } else {
                $careerApplication->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several careerApplications by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            CareerApplication::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            CareerApplication::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $careerApplication = CareerApplication::withTrashed()->find($id);

        if ($careerApplication){
            $careerApplication->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several careerApplications by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $careerApplication = CareerApplication::whereIn('id', $request->ids)->onlyTrashed();
        if ($careerApplication){
            $careerApplication->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
