<?php

namespace App;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property mixed id
 * @property mixed tags_ids
 */
class Article extends Model
{
    //use soft delete
    use SoftDeletes;

    protected $fillable = [
        'image',
        'category_id',
        'tags_ids',
    ];

    protected $casts = ['tags_ids' => 'array'];

    /**
     * Return Main Photo of article
     * @return string|null
     */
    public function getMainPhotoAttribute() : ?string
    {
        $mainPhoto= $this->image ?? null;
        if( !$mainPhoto ){
            $photo= $this->photos()->first();
            $mainPhoto = $photo ? $photo->image : null;
        }
        $mainPhoto = getImage($mainPhoto);
        return $mainPhoto;
    }

    /**
     * Return Slug of article
     * @return string|null
     */
    public function getSlugTextAttribute() : ?string
    {
        return str_slug(strtolower($this->currentDescription->slug));
    }

    /**
     * Return Url of article
     * @return string|null
     */
    public function getUrlAttribute() : ?string
    {
        return route('articles.details',[ 'article'=> $this->id, 'slug'=> $this->slug_text]);
    }

    /**
     * Return Date of article
     * @return string|null
     */
    public function getDateAttribute() : ?string
    {
        return Carbon::parse($this->updated_at)->format('F d Y');
    }

    /**
     * Return current descriptions of article
     * @return HasOne
     */
    public function currentDescription() : HasOne
    {
        return $this->hasOne(ArticleDescription::class)->where('language_id', currentLanguage()->id);
    }

    /**
     * Return descriptions for article
     * @return HasMany
     */
    public function descriptions() : HasMany
    {
        return $this->hasMany(ArticleDescription::class);
    }



    /**
     * Return other photos for article
     * @return HasMany
     */
    public function photos () :HasMany
    {
        return $this->hasMany(ArticlePhoto::class);
    }


    public static function withDescription($ids = [], $limit = null, $skip = null, $random = false)
    {
        $langId = currentLanguage()->id;
        $query = self::orderByDesc('articles.id')
            ->join('article_descriptions as aDesc', 'aDesc.article_id', 'articles.id')
            ->where('aDesc.language_id', $langId);
        if (count($ids)) {
            $query->whereIn('articles.id', $ids);
        }

        if ($limit){
            $query->limit($limit);
        }

        if ($skip){
            $query->skip($skip);
        }

        if ($random){
            $query->inRandomOrder();
        }

        $query->select([
            'articles.id',
            'articles.created_at',
            'articles.image',
            'aDesc.title',
            'aDesc.description',
        ]);

        return count($ids) == 1 ? $query->first() : $query->get();
    }



}
